--[[=============================================================================
    Commands received from the AVSwitch proxy (ReceivedFromProxy)

    Copyright 2023 Key Digital Systems. All Rights Reserved.
===============================================================================]]

-- This macro is utilized to identify the version string of the driver template version used.
if (TEMPLATE_VERSION ~= nil) then
	TEMPLATE_VERSION.device_messages = "2015.03.31"
end

--=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
-- Power Functions
--=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
--[[
	Proxy Command: CONNECT_OUTPUT
	Parameters:
		output:			mod 1000 value of Output Connection id
		class:			connection class
		output_id:		value of Output Connection id
--]]
function CONNECT_OUTPUT(output, class, output_id)    
    local command_delay = tonumber(Properties["Power On Delay Seconds"])
    local delay_units = "SECONDS"
    local command
    if (gControlMethod == "IR") then
		-- Switcher does not support Zone On. Leave cmd blank
		command = CMDS_IR_ZONES[output]["CONNECT_OUTPUT"]
		LogTrace("command = " .. command)
		PackAndQueueCommand("CONNECT_OUTPUT", command, command_delay, delay_units)
    else

		-- Switcher does not support Zone On. Leave cmd blank
		command = ""
		LogTrace("command = " .. command)
		PackAndQueueCommand("CONNECT_OUTPUT", command, command_delay, delay_units)
		
    end
end

--[[
	Proxy Command: DISCONNECT_OUTPUT
	Parameters:
		output:			mod 1000 value of Output Connection id
		class:			connection class
		output_id:		value of Output Connection id
--]]
function DISCONNECT_OUTPUT(output, class, output_id)
    local command_delay = tonumber(Properties["Power Off Delay Seconds"])
    local delay_units = "SECONDS"
    local command 
    local isAudio = (output_id >= 4000)
    
    if(isAudio) then
	   LogTrace("Received From Proxy: DISCONNECT_OUTPUT for AUDIO:: Output(" .. output_id ..  ")")   
    else
	   LogTrace("Received From Proxy: DISCONNECT_OUTPUT for VIDEO:: Output(" .. output_id ..  ")") 
    end

    --[[In certain scenarios (based upon AV connections and project bindings), a DISCONNECT_OUTPUT command is sent in error by the proxy.
	   This code block utilized timers that are started in the SET_INPUT function to determine if this DISCONNECT_OUTPUT command is
	   associated with a reselection transaction, in whcih case we will abort...
    --]]
    if not (isAudio) then
	   if (isAudioSelectionInProgress(output_id) == true) then 
		  LogTrace("Audio Selection is in progress.  Not Disconnecting.")
		  return 
	   elseif (isHDMIAudioSelectionInProgress(output, output_id) == true) then 
		  LogTrace("HDMI Audio Selection is in progress.  Not Disconnecting.")
		  return  
	   end
    end    		
    
    -- Switcher does not support Zone Off. Leave cmd blank
    if (gControlMethod == "IR") then
	   command = CMDS_IR_ZONES[output]["DISCONNECT_OUTPUT"]
    else		
	   if (isAudio) then
		  gOutputToInputAudioMap[output] = -1
		  command = ""  	--audio disconnect element
	   else
		  gOutputToInputMap[output] = -1
		  command = ""		--video disconnect element
	   end

	   command = command .. ""
    end  
    
    LogTrace("command = " .. command)
    PackAndQueueCommand("DISCONNECT_OUTPUT", command, command_delay, delay_units)    
		
end

--=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
-- Input Selection and AV Path Functions
--=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
--[[
	Proxy Command: SET_INPUT
	Parameters:
		idBinding:			proxybindingid of proxy bound to input connection
		output: 			mod 1000 value of Output Connection ID	
		input: 				mod 1000 value of Input Connection ID
		input_id:			Input Connection ID
		class:				Connection Class
		output_id:			Output Connection id
		bSwitchSeparate		Setting of can_switch_separately capability
		bVideo				indicates a video path selection
		bAudio				indicates an audio path selection
--]]
function SET_INPUT(idBinding, output, input, input_id, class, output_id, bSwitchSeparate, bVideo, bAudio)
    local command
	-- Override 'bSwitchSeparate' flag to true. KD switchers can always switch separately. TODO Fix C4 Bug - Figure out why 'bSwitchSeparate' is null
	bSwitchSeparate = true
    if (bSwitchSeparate) then	--device switches audio and video separatley
    
	   --DEBOUNCE LOGIC - in some instances (based upon signal type and project bindings) redundant commands are sent from the proxy
	   --so we can test here and abort the command, if desired
	   --if (gAVSwitchProxy._PreviouslySelectedInput[output_id] == input_id) then return end    
	   
	   local isAudio = (output_id >= 4000)
	   if (isAudio) then
		  startAudioSelectionTimer(output)		--used in DISCONNECT_OUTPUT
		  gOutputToInputAudioMap[output] = input
		  command = string.format("SPO%04dSA%04d", KD_Helper:ConvertToKDID(output), KD_Helper:ConvertToKDID(input))
	   else
		  startHDMIAudioSelectionTimer(output)	--used in DISCONNECT_OUTPUT
		  local mirrored_output_id = getMirroredOutputID(output_id)
		  local mirrorState = getMirroredOutputState(output_id)
		  if (mirrorState == "NO MIRROR ZONE") or (mirrorState == "AUDIO ZONE WITH MIRRORED VIDEO ZONE") or (mirrorState == "VIDEO ZONE WITH MIRRORED AUDIO ZONE") then
			 gOutputToInputMap[output] = input
			 command = string.format("SPO%04dSI%04d", KD_Helper:ConvertToKDID(output), KD_Helper:ConvertToKDID(input))
		  else
			 LogError("SET_INPUT: ERROR - mirrorState is unknown")
			 return
		  end
		  
	   end
	   
    else	--[[device does not switch audio and video separatley, so we can use the mod 1000 values of the connection id 
			 to determine the audio and video legs of a given connection --]]
	   
	   --[[DEBOUNCE LOGIC - in some instances (based upon signal type and project bindings) redundant commands are sent from the proxy
		  so we can test here and abort the command, if desired --]]
	   --if (gAVSwitchProxy._PreviouslySelectedInput[output] == input) then return end 

	   if (gControlMethod == "IR") then		
		-- Send IR
		command = tInputCommandMap_IR[output][tInputConnMapByID[input].Name]	
	   else
		-- Send AV switch command
		command = string.format("SPO%04dSI%04d", KD_Helper:ConvertToKDID(output), KD_Helper:ConvertToKDID(input))
	   end 	

    end
	
    LogTrace("command = " .. command)
    PackAndQueueCommand("SET_INPUT", command)

end

--=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
-- Volume Control Functions
--		Proxy control will affect all audio outputs: HDMI and External
--		Audio. The proxy commands will only do all or nothing. No separate
--		controls. All separate controls can be found in the device specific
--		commands send from execute commands section
--=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
--[[
	Proxy Command: MUTE_OFF
	Parameters:
		output:		mod 1000 value of Output Connection id	
--]]
function MUTE_OFF(output)
	local command
	if (gControlMethod == "IR") then	
		command = CMDS_IR_ZONES[output]["MUTE_OFF"]	
		LogTrace("command = " .. command)
		PackAndQueueCommand("MUTE_OFF", command)
	else
		-- Send Video Mute off, aka Output Enable
		command = string.format("SPO%04dON", KD_Helper:ConvertToKDID(output))
		LogTrace("command = " .. command)
		PackAndQueueCommand("MUTE_OFF", command)
	end 		
end

--[[
	Proxy Command: MUTE_ON
	Parameters:
		output:		mod 1000 value of Output Connection id	
--]]
function MUTE_ON(output)
	local command
	if (gControlMethod == "IR") then	
		command = CMDS_IR_ZONES[output]["MUTE_ON"]	
		LogTrace("command = " .. command)
		PackAndQueueCommand("MUTE_ON", command)
	else
		-- Send Video Mute on, aka Output Disable
		command = string.format("SPO%04dOFF", KD_Helper:ConvertToKDID(output))
		LogTrace("command = " .. command)
		PackAndQueueCommand("MUTE_ON", command)
	end 		
end

--[[
	Proxy Command: MUTE_TOGGLE
	Parameters:
		output:		mod 1000 value of Output Connection id	
--]]
function MUTE_TOGGLE(output)
	LogError("MUTE_TOGGLE: ERROR - Not supported")
end

--[[
	Proxy Command: SET_VOLUME_LEVEL
	Parameters:
		output:			mod 1000 value of Output Connection id	
		c4VolumeLevel:	volume level to be set represented in Control4 scale (0-100)
--]]
function SET_VOLUME_LEVEL(output, c4VolumeLevel)
	LogError("SET_VOLUME_LEVEL: ERROR - Not supported")
end

--[[
	Helper Function: SET_VOLUME_LEVEL_DEVICE
	Parameters:
		output:				mod 1000 value of Output Connection id	
		deviceVolumeLevel:	volume level to be set represented in device scale (as sepcified in the device's control protocol)
--]]
function SET_VOLUME_LEVEL_DEVICE(output, deviceVolumeLevel)
	LogError("SET_VOLUME_LEVEL_DEVICE: ERROR - Not supported")
end

--[[
	Proxy Command: PULSE_VOL_DOWN
	Parameters:
		output: mod 1000 value of Output Connection id	
--]]
function PULSE_VOL_DOWN(output)
	LogError("PULSE_VOL_DOWN: ERROR - Not supported")
end

--[[
	Proxy Command: PULSE_VOL_UP
	Parameters:
		output: mod 1000 value of Output Connection id	
--]]
function PULSE_VOL_UP(output)
	LogError("PULSE_VOL_UP: ERROR - Not supported")
end

--=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
-- Helper Functions
--=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=

--[[
	Helper Function: SEND_COMMAND_FROM_COMMAND_TABLE
	Parameters:
		idBinding: proxy id	
		output: mod 1000 value of Output Connection id
		command_name: name of command to be sent
--]]
function SEND_COMMAND_FROM_COMMAND_TABLE(idBinding, output, command_name)
    local output_for_log = output or "nil"
    LogTrace("SEND_COMMAND_FROM_COMMAND_TABLE(), idBinding=" .. idBinding .. ", output=" .. output_for_log .. ", command_name=" .. command_name)
	
	local command = GetCommandFromCommandTable(idBinding, output, command_name)
	
	if (command == nil) then
		LogTrace("command is nil")
	else
		LogTrace("command = " .. command)
		PackAndQueueCommand(command_name, command)
	end			
end

--[[
	Helper Function: GetCommandFromCommandTable
	Parameters:
		idBinding: proxy id	
		output: mod 1000 value of Output Connection id
		command_name: name of command to be returned
--]]
function GetCommandFromCommandTable(idBinding, command_name)
	LogTrace("GetCommand()")
	local t = {}
	
	if (gControlMethod == "IR") then
		t = CMDS_IR
	else
		t = CMDS
	end	

	if (t[idBinding] ~= nil) then
	   if (t[idBinding][command_name] ~= nil) then
		  return t[idBinding][command_name]
	   end
	end
	
	if (t[command_name] ~= nil) then
		return t[command_name]
	else
		LogWarn('GetCommandFromCommandTable: command not defined - '.. command_name)
		return nil
	end	
	
end


--[[=============================================================================
    Helper Functions
===============================================================================]]
function GetDeviceStatus()
    LOG:Trace("GetDeviceStatus()")
	local command = GetCommandFromCommandTable(0, "STATUS")
	LOG:Trace("command = " .. command)
	PackAndQueueCommand("GetDeviceStatus: ", command)	
end
