--[[=============================================================================
    Get, Handle and Dispatch message functions

    Copyright 2023 Key Digital Systems. All Rights Reserved.
===============================================================================]]

-- This macro is utilized to identify the version string of the driver template version used.
if (TEMPLATE_VERSION ~= nil) then
	TEMPLATE_VERSION.device_messages = "2015.03.31"
end

--[[=============================================================================
    GetMessage()
  
    Description:
    Used to retrieve a message from the communication buffer. Each driver is
    responsible for parsing that communication from the buffer.
  
    Parameters:
    None
  
    Returns:
    A single message from the communication buffer
===============================================================================]]
function GetMessage()
	local message, pos
	
	-- All messages are terminated by carraige return and linefeed <CR><LF>
	local pattern = "^(.-)\r\n()"
	
	if (gReceiveBuffer:len() > 0) then
		message, pos = string.match(gReceiveBuffer, pattern)
		if (message == nil) then
			--LOG:Info("Do not have a complete message")
			return ""
		end
		gReceiveBuffer = gReceiveBuffer:sub(pos)		
	end

	return message
	
end

--[[=============================================================================
    HandleMessage(message)]

    Description
    This is where we parse the messages returned from the GetMessage()
    function into a command and data. The call to 'DispatchMessage' will use the
    'name' variable as a key to determine which handler routine, function, should
    be called in the DEV_MSG table. The 'value' variable will then be passed as
    a string parameter to that routine.

    Parameters
    message(string) - Message string containing the function and value to be sent to
                      DispatchMessage

    Returns
    Nothing
===============================================================================]]
function HandleMessage(message)
	if (type(message) == "table") then
		LogTrace("HandleMessage. Message is ==>")
		LogTrace(message)
		LogTrace("<==")
	else
		LogTrace("HandleMessage. Message is ==>%s<==", message)
	end

	-- Handle Single Output Status
	local output, input, pos = string.match(message, "<s>SPO(%d+)SI(%d+)</s>()")
	if (output ~= nil and input ~= nil) then
		DispatchMessage("INPUT_VIDEO", tonumber(output), tonumber(input))
		return
	end

	-- Handle All Output Status
	local input, pos = string.match(message, "<s>SPOASI(%d+)</s>()")
	if (input ~= nil) then
		for i = 1, (G_KD_OUTPUT_TOTAL+1), 1
		do
			DispatchMessage("INPUT_VIDEO", tonumber(i), tonumber(input))
		end
		return
	end

	--
	-- Special handling. We need to get output status of multi line outputs
	--

	-- Handle video feedback in STAA response
	local tVal1, tVal2, pos = string.match(message, "%-%- (.+) Video Output Status (.+)%-%-()")
	if (tVal1 ~= nil and tVal2 ~= nil) then
		LogDebug("STAA Parser Started")
		G_KD_FEEDBACK_OUTPUT_STATUS = true
		return
	end

	-- Only record STAA output when we're in the output status
	if (G_KD_FEEDBACK_OUTPUT_STATUS) then

		-- Get output number in STAA response
		local output, tVal1, pos = string.match(message, "%-%- (%d+): (.+)%-%-()")
		if (output ~= nil and tVal1 ~= nil) then
			G_KD_FEEDBACK_OUTPUT_VAL = tonumber(output)
			return
		end

		-- Only record output status when we have a valid output number
		if (G_KD_FEEDBACK_OUTPUT_VAL ~= 0) then
			-- Get input status in STAA response
			local tVal1, input, mute, tVal2, pos = string.match(message, "%-%- (.+) IN=(%d+), OUT=(%a%a)(.+)%-%-()")
			if (input ~= nil and mute ~= nil and tVal1 ~= nil and tVal2 ~= nil) then
				DispatchMessage("INPUT_VIDEO", tonumber(G_KD_FEEDBACK_OUTPUT_VAL), tonumber(input))
				DispatchMessage("MUTE", tonumber(G_KD_FEEDBACK_OUTPUT_VAL), mute)

				-- We got what we need. Reset output number
				G_KD_FEEDBACK_OUTPUT_VAL = 0
				return
			end
			
		end

		-- Figure out where to end STAA output response. Find the line with all dashes
		local tVal1, pos = string.match(message, "%-%-%-%-%-%-%-%-%-%-%-(.+)%-%-%-%-%-%-%-%-%-%-%-%-%-%-%-%-%-()")
		if (tVal1 ~= nil) then
			LogDebug("STAA Parser Ended")
			G_KD_FEEDBACK_OUTPUT_STATUS = false
			G_KD_FEEDBACK_OUTPUT_VAL = 0
			return
		end

	end
	
end

--[[=============================================================================
    DispatchMessage(MsgKey, MsgData)

    Description
    Parse routine that will call the routines to handle the information returned
    by the connected system.

    Parameters
    MsgKey(string)  - The function to be called from within DispatchMessage
    MsgData(string) - The parameters to be passed to the function found in MsgKey

    Returns
    Nothing
===============================================================================]]
function DispatchMessage(MsgKey, ...)
	if (DEV_MSG[MsgKey] ~= nil and (type(DEV_MSG[MsgKey]) == "function")) then
		LogInfo("DEV_MSG." .. tostring(MsgKey) .. ":  " .. table.concat({...},", "))
		local status, err = pcall(DEV_MSG[MsgKey], ...)
		if (not status) then
			LogError("LUA_ERROR: " .. err)
		end
	else
		LogTrace("HandleMessage: Unhanded command = " .. MsgKey)
	end
end


--[[=============================================================================
    Our DEV_MSG Functions
===============================================================================]]

function DEV_MSG.INPUT_VIDEO(output, input)
	LogInfo("INPUT_OUTPUT_VIDEO_CHANGED, output = " .. output .. ", input = " .. input)
	-- AV changed. Update audio status
	gAVSwitchProxy:dev_InputOutputChanged(KD_Helper:ConvertToInputConnectionID(input), KD_Helper:ConvertToVideoConnectionID(output))
end

function DEV_MSG.MUTE(output, mute)
	LogInfo("OUTPUT_MUTE_CHANGED, output = " .. output .. ", mute = " .. mute)
	-- Video mute changed
	if (mute == "OF") then 
		state = "True"
	elseif (mute == "ON") then
		state = "False"
	else
		LogWarn("DEV_MSG.MUTE(), value not valid, exiting...")
		return
	end
	gAVSwitchProxy:dev_MuteChanged(KD_Helper:ConvertToZeroBasedID(output), state)
end
