--[[=============================================================================
    Key Digital Switcher Template for AVSwitch Driver

    Copyright 2023 Key Digital Systems. All Rights Reserved.
===============================================================================]]
require "common.c4_driver_declarations"
require "common.c4_common"
require "common.c4_init"
require "common.c4_property"
require "common.c4_command"
require "common.c4_notify"
require "common.c4_network_connection"
require "common.c4_serial_connection"
require "common.c4_ir_connection"
require "common.c4_utils"
require "lib.c4_timer"
require "actions"
require "device_specific_commands"
require "device_messages"
require "avswitch_init"
require "properties"
require "proxy_commands"
require "connections"
require "avswitch.avswitch_proxy_class"
require "avswitch.avswitch_proxy_commands"
require "avswitch.avswitch_proxy_notifies"
require "av_path"
require "kd_helper"


-- This macro is utilized to identify the version string of the driver template version used.
if (TEMPLATE_VERSION ~= nil) then
	TEMPLATE_VERSION.driver = "2015.03.31"
end

--[[=============================================================================
    Constants
===============================================================================]]
AVSWITCH_PROXY_BINDINGID = 5001

--[[
    Number of Inputs/Outputs 0 based
--]]
G_KD_INPUT_TOTAL = 63
G_KD_OUTPUT_TOTAL = 63

--[[
    Device Volume Range
--]]
MIN_DEVICE_LEVEL = 0
MAX_DEVICE_LEVEL = 100	


--[[=============================================================================
    Globals
===============================================================================]]
G_KD_FEEDBACK_OUTPUT_STATUS = false
G_KD_FEEDBACK_OUTPUT_VAL = 0


--[[=============================================================================
    Initialization Code
===============================================================================]]
function ON_DRIVER_EARLY_INIT.main()
	
end

function ON_DRIVER_INIT.main()

end

function ON_DRIVER_LATEINIT.main()
    C4:urlSetTimeout (20) 
    DRIVER_NAME = C4:GetDriverConfigInfo("name")
	
	SetLogName(DRIVER_NAME)
end

function ON_DRIVER_EARLY_INIT.avswitch_driver()

end

function ON_DRIVER_INIT.avswitch_driver()
    -- Map to track full output id to Room id. This will be used in DISCONNECT_OUTPUT
    --to prevent the zone  from turning off when we are just selecting an alternate audio source
    --entries will be seeded in AV_OUTPUT_TO_INPUT_VALID
    gVideoProviderToRoomMap = {}
    gAudioProviderToRoomMap = {}

    --entries will be seeded in AV_OUTPUT_TO_INPUT_VALID
    gLastReportedAVPaths = {}
	
    --Connection Type 
    gAVPathType = {
	   [5] = "VIDEO",
	   [6] = "AUDIO",
	   [7] = "ROOM",
    }
    
    -- Loop through all output and initialize it
    local tVolumeRamping = {}
    gOutputToInputMap = {}
    gOutputToInputAudioMap = {}
    for i = 0, G_KD_OUTPUT_TOTAL, 1
    do
        -- No Ramping available
        tVolumeRamping[i] = {state = false,mode = "",}
	
	    -- Maps to keep track of inputs and outputs.  Useful for EDID management logic...
        gOutputToInputMap[i] = -1
        gOutputToInputAudioMap[i] = -1

        --  Dynamically create Audio/HdmiAudio Timers
        local AudioInProgressTimerCmd = string.format("gAudioSelectionInProgressOutput%dTimer = c4_timer:new('AudioSelectionInProgressOutput%dTimer', 500, 'MILLISECONDS', OnAudioSelectionInProgressOutputTimerExpired, false, %d)", i, i, i)
        local HdmiAudioInProgressTimerCmd = string.format("gHDMIAudioSelectionInProgressOutput%dTimer = c4_timer:new('HDMIAudioSelectionInProgressOutput%dTimer', 500, 'MILLISECONDS', OnHDMIAudioSelectionInProgressOutputTimerExpired, false, %d)", i, i, i)
        KD_Helper:CallStatement(AudioInProgressTimerCmd)
        KD_Helper:CallStatement(HdmiAudioInProgressTimerCmd)
    end
	
    -- Create an instance of the AVSwitchProxy class
    local  bProcessesDeviceMessages = true
    local bUsePulseCommandsForVolumeRamping = false
    if (Properties["Ramping Method"] == "Pulse Commands") then bUsePulseCommandsForVolumeRamping = true end
    gAVSwitchProxy = AVSwitchProxy:new(AVSWITCH_PROXY_BINDINGID, bProcessesDeviceMessages, tVolumeRamping, bUsePulseCommandsForVolumeRamping)

    local minDeviceLevel = MIN_DEVICE_LEVEL
    local maxDeviceLevel = MAX_DEVICE_LEVEL
    tVolumeCurve = getVolumeCurve(minDeviceLevel, maxDeviceLevel)
	
	--[[
	For the "Volume Curve" method, tVolumeCurve is used to store volume level values that will be used to build volume commands during volume ramping. Specifically, they are used in GetNextVolumeCurveValue() which is called from the ContinueVolumeRamping() function.  Property values for "Volume Ramping Steps" and "Volume Ramping Slope" can be adjusted to get a smooth volume ramping from low to high volume.
	--]]
end

function getVolumeCurve(minDeviceLevel, maxDeviceLevel)
    local steps = tonumber(Properties["Volume Ramp Steps"])
    local slope = tonumber(Properties["Volume Ramp Slope"])
    local tCurve = gAVSwitchProxy:CreateVolumeCurve(steps, slope, minDeviceLevel, maxDeviceLevel)
    
    return tCurve 
end

--[[=============================================================================
    Driver Code
===============================================================================]]
function PackAndQueueCommand(...)
    local command_name = select(1, ...) or ""
    local command = select(2, ...) or ""
    local command_delay = select(3, ...) or tonumber(Properties["Command Delay Milliseconds"])
    local delay_units = select(4, ...) or "MILLISECONDS"
    LogTrace("PackAndQueueCommand(), command_name = " .. command_name .. ", command delay set to " .. command_delay .. " " .. delay_units)
    if (command == "") then
	   LogWarn("PackAndQueueCommand(), command_name = " .. command_name .. ", command string is empty - exiting PackAndQueueCommand()")
	   return
    end
	
	-- Pack command with any any required starting or ending characters
    local cmd, stx, etx
    if (gControlMethod == "Network") then
		stx = ""
		etx = "\r"
		cmd = stx .. command .. etx
    elseif (gControlMethod == "Serial") then
		stx = ""
		etx = "\r"
		cmd = stx .. command .. etx
    elseif (gControlMethod == "IR") then
		cmd = command
    else
		LogWarn("PackAndQueueCommand(): gControlMethod is not valid, ".. gControlMethod)
		return
    end
    gCon:QueueCommand(cmd, command_delay, delay_units, command_name)	
	
end
