--[[=============================================================================
    Key Digital - Lua Helper - Bunch of helpful functions for driver development

    Copyright 2024 Key Digital Systems. All Rights Reserved.
===============================================================================]]

--[[=============================================================================
    KD_Helper()
  
    Description:
    KD Helper class for some OOD action
===============================================================================]]
KD_Helper = {}

--[[=============================================================================
    IsNumber()
  
    Description:
    Checks if the input parameter is a valid number
  
    Parameters:
    param           Value to check
  
    Returns:
    bool            True if param is a number. False otherwise
===============================================================================]]
function KD_Helper:IsNumber(param)
    return (type(param) == "number")
end

--[[=============================================================================
    IsString()
  
    Description:
    Checks if the input parameter is a valid string
  
    Parameters:
    param           Value to check
  
    Returns:
    bool            True if param is a string. False otherwise
===============================================================================]]
function KD_Helper:IsString(param)
    return (type(param) == "string")
end

--[[=============================================================================
    InRange()
  
    Description:
    Checks if the input parameter is within (including) the min and max range
  
    Parameters:
    param           Value to check
    min             Minimum value
    max             Maximum value
  
    Returns:
    bool            True if param is within range. False otherwise
===============================================================================]]
function KD_Helper:InRange(param, min, max)
    return ((self:IsNumber(param)) and (param >= min) and (param <= max))
end

--[[=============================================================================
    CallFunctionByName()
  
    Description:
    Dynamically calls a function by name using the provided param string
  
    Parameters:
    param           Function Name
  
    Returns:
    The original function's return value
===============================================================================]]
function KD_Helper:CallFunctionByName(param)
    -- Validate
    if not self:IsString(param) then
        return false
    end

    -- Build cmd
    local stringCmd = string.format("return %s", param)

    -- Call function by name
    LogDebug("KD_Helper:CallFunctionByName() loadstring = '" .. stringCmd .. "'")
    local loadedCmd = assert(loadstring(stringCmd))
    return loadedCmd()
end

--[[=============================================================================
    CallStatement()
  
    Description:
    Dynamically calls a statement using the provided param string
  
    Parameters:
    param           Statement
  
    Returns:
    bool            True if statement succeeded. Throws if it failed
===============================================================================]]
function KD_Helper:CallStatement(param)
    -- Validate
    if not self:IsString(param) then
        return false
    end

    -- Call entire statement dynamically
    LogDebug("KD_Helper:CallStatement() loadstring = '" .. param .. "'")
    assert(loadstring(param))()
    return true
end

--[[=============================================================================
    ConvertToZeroBasedID()
  
    Description:
    Takes an input value and converts it to its 0 based value by subtracting 1
  
    Parameters:
    param           ID
  
    Returns:
    bool            The ID minus 1
===============================================================================]]
function KD_Helper:ConvertToZeroBasedID(id)
	return id - 1
end

--[[=============================================================================
    ConvertToInputConnectionID()
  
    Description:
    Takes an input value and converts it to its 0 based Input Connection ID
    Number by adding 1000
  
    Parameters:
    param           ID
  
    Returns:
    bool            The ID plus 1000 minus 1
===============================================================================]]
function KD_Helper:ConvertToInputConnectionID(id)
	return self:ConvertToZeroBasedID(id + 1000)
end

--[[=============================================================================
    ConvertToVideoConnectionID()
  
    Description:
    Takes an input value and converts it to its 0 based Video Connection ID
    Number by adding 2000
  
    Parameters:
    param           ID
  
    Returns:
    bool            The ID plus 2000 minus 1
===============================================================================]]
function KD_Helper:ConvertToVideoConnectionID(id)
	return self:ConvertToZeroBasedID(id + 2000)
end

--[[=============================================================================
    ConvertToAudioConnectionID()
  
    Description:
    Takes an input value and converts it to its 0 based Audio Connection ID
    Number by adding 4000
  
    Parameters:
    param           ID
  
    Returns:
    bool            The ID plus 4000 minus 1
===============================================================================]]
function KD_Helper:ConvertToAudioConnectionID(id)
	return self:ConvertToZeroBasedID(id + 4000)
end

--[[=============================================================================
    ConvertToKDID()
  
    Description:
    Takes an input value and converts it to its Key Digital ID by adding 1
  
    Parameters:
    param           ID
  
    Returns:
    bool            The ID plus 1
===============================================================================]]
function KD_Helper:ConvertToKDID(id)
	return id + 1
end