--[[=============================================================================
    Get, Handle and Dispatch message functions

    Copyright 2023 Key Digital Systems. All Rights Reserved.
===============================================================================]]

-- This macro is utilized to identify the version string of the driver template version used.
if (TEMPLATE_VERSION ~= nil) then
	TEMPLATE_VERSION.device_messages = "2015.03.31"
end

--[[=============================================================================
    GetMessage()
  
    Description:
    Used to retrieve a message from the communication buffer. Each driver is
    responsible for parsing that communication from the buffer.
  
    Parameters:
    None
  
    Returns:
    A single message from the communication buffer
===============================================================================]]
function GetMessage()
	local message, pos
	
	-- All messages are terminated by carraige return and linefeed <CR><LF>
	local pattern = "^(.-)\r\n()"
	
	if (gReceiveBuffer:len() > 0) then
		message, pos = string.match(gReceiveBuffer, pattern)
		if (message == nil) then
			--LOG:Info("Do not have a complete message")
			return ""
		end
		gReceiveBuffer = gReceiveBuffer:sub(pos)		
	end

	return message
	
end

--[[=============================================================================
    HandleMessage(message)]

    Description
    This is where we parse the messages returned from the GetMessage()
    function into a command and data. The call to 'DispatchMessage' will use the
    'name' variable as a key to determine which handler routine, function, should
    be called in the DEV_MSG table. The 'value' variable will then be passed as
    a string parameter to that routine.

    Parameters
    message(string) - Message string containing the function and value to be sent to
                      DispatchMessage

    Returns
    Nothing
===============================================================================]]
function HandleMessage(message)
	if (type(message) == "table") then
		LogTrace("HandleMessage. Message is ==>")
		LogTrace(message)
		LogTrace("<==")
	else
		LogTrace("HandleMessage. Message is ==>%s<==", message)
	end

	-- Handle Output Statuses
	local output, command, value, pos = string.match(message, "<s>SPO(%d%d)(%a%a)(.+)</s>()")
	if (output ~= nil and command ~= nil and value ~= nil) then
		command = string.upper(command)
		-- Audio only status
		if (command == "SA" and tonumber(value) ~= nil) then
			DispatchMessage("INPUT_AUDIO", tonumber(output), tonumber(value))
			return

		-- Video only status
		elseif (command == "SB" and tonumber(value) ~= nil) then
			DispatchMessage("INPUT_VIDEO", tonumber(output), tonumber(value))
			return

		-- Audio/Video Status
		elseif (command == "SI" and tonumber(value) ~= nil) then
			DispatchMessage("INPUT_AUDIO_VIDEO", tonumber(output), tonumber(value))
			return
		
		-- All the mutes
		elseif (command == "HA" or command == "AA" or command == "DA") then
			DispatchMessage("MUTE", tonumber(output), value)
			return

		end
	end

	-- Handle video feedback in STA response
	local output, input, link, debug, hdmiAudio, pos = string.match(message, "-- (%d%d): Input = (%d%d), Link = (%a+), DBG = (%a+), HA = (%a+)()")
	if (output ~= nil and input ~= nil and link ~= nil and debug ~= nil and hdmiAudio ~= nil) then
		DispatchMessage("INPUT_VIDEO", tonumber(output), tonumber(input))
		DispatchMessage("MUTE", tonumber(output), hdmiAudio)
		return
	end

	-- Handle audio feedback in STA response
	local output, input, analog, optical, pos = string.match(message, "-- (%d%d) : Input = (%d%d), Analog Balanced = (%a+), Optical = (%a+)()")
	if (output ~= nil and input ~= nil and analog ~= nil and optical ~= nil) then
		DispatchMessage("INPUT_AUDIO", tonumber(output), tonumber(input))
		DispatchMessage("MUTE", tonumber(output), analog)
		DispatchMessage("MUTE", tonumber(output), optical)
		return
	end
	
end

--[[=============================================================================
    DispatchMessage(MsgKey, MsgData)

    Description
    Parse routine that will call the routines to handle the information returned
    by the connected system.

    Parameters
    MsgKey(string)  - The function to be called from within DispatchMessage
    MsgData(string) - The parameters to be passed to the function found in MsgKey

    Returns
    Nothing
===============================================================================]]
function DispatchMessage(MsgKey, ...)
	if (DEV_MSG[MsgKey] ~= nil and (type(DEV_MSG[MsgKey]) == "function")) then
		LogInfo("DEV_MSG." .. tostring(MsgKey) .. ":  " .. table.concat({...},", "))
		local status, err = pcall(DEV_MSG[MsgKey], ...)
		if (not status) then
			LogError("LUA_ERROR: " .. err)
		end
	else
		LogTrace("HandleMessage: Unhanded command = " .. MsgKey)
	end
end


--[[=============================================================================
    Our DEV_MSG Functions
===============================================================================]]

function DEV_MSG.INPUT_AUDIO_VIDEO(output, input)
	LogInfo("INPUT_OUTPUT_AV_CHANGED, output = " .. output .. ", input = " .. input)
	-- AV changed. Update video and audio statuses
	gAVSwitchProxy:dev_InputOutputChanged(ConvertToInputConnectionID(input), ConvertToVideoConnectionID(output))
	gAVSwitchProxy:dev_InputOutputChanged(ConvertToInputConnectionID(input), ConvertToAudioConnectionID(output))
end

function DEV_MSG.INPUT_AUDIO(output, input)
	LogInfo("INPUT_OUTPUT_AUDIO_CHANGED, output = " .. output .. ", input = " .. input)
	-- AV changed. Update audio status
	gAVSwitchProxy:dev_InputOutputChanged(ConvertToInputConnectionID(input), ConvertToAudioConnectionID(output))
end

function DEV_MSG.INPUT_VIDEO(output, input)
	LogInfo("INPUT_OUTPUT_VIDEO_CHANGED, output = " .. output .. ", input = " .. input)
	-- AV changed. Update audio status
	gAVSwitchProxy:dev_InputOutputChanged(ConvertToInputConnectionID(input), ConvertToVideoConnectionID(output))
end

function DEV_MSG.MUTE(output, mute)
	LogInfo("OUTPUT_MUTE_CHANGED, output = " .. output .. ", mute = " .. mute)
	-- Audio mute changed
	muteValue = string.upper(string.sub(mute, 1, 1))
	if (muteValue == "D") then 
		state = "True"
	elseif (muteValue == "E") then
		state = "False"
	else
		LogWarn("DEV_MSG.MUTE(), value not valid, exiting...")
		return
	end
	gAVSwitchProxy:dev_MuteChanged(ConvertToZeroBasedID(output), state)
end


--[[=============================================================================
    Helper Functions
===============================================================================]]
function ConvertToZeroBasedID(id)
	return id - 1
end
function ConvertToInputConnectionID(id)
	return ConvertToZeroBasedID(id + 1000)
end
function ConvertToVideoConnectionID(id)
	return ConvertToZeroBasedID(id + 2000)
end
function ConvertToAudioConnectionID(id)
	return ConvertToZeroBasedID(id + 4000)
end
